/**
* \file: automounter_api_info.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#ifndef AUTOMOUNTER_API_INFO_H_
#define AUTOMOUNTER_API_INFO_H_

#include "automounter_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief Used to get a snapshot of devices and partitions currently managed by the automounter.
 *
 * The application can use this function to get the current state and other information about devices and partitions
 * currently managed by the automounter. This function returns immediately.
 *
 * The information about the devices and partitions reaches the applications asynchronously. For each device and
 * partition that fits to the request, the callback function registered with the events on_update_device_info and
 * on_update_partition_info respectively are called by the shared library. See the API description of the
 * automounter_api_callbacks_t structure for more details.
 *
 * The application can limit the amount of information by setting the scope parameter. See the description of
 * the snapshot_scope_t enumeration for more details.
 *
 * The sequence of callback calls ends with a call of the callback registered with the on_snapshot_complete event to
 * signal the application that the snapshot request has been processed completely.
 *
 * A pointer to an integer can be passed to this function call to get an id for this request. This id is passed to each
 * of the involved callbacks, which enables the application to identify the callback calls that belong to a certain
 * request. In most cases, applications are calling this function only ones at a time so that they don't need an
 * identifier for their request. In this cases, NULL can be passed to the function.
 *
 * \param scope the scope of the snapshot request
 * \param request_id_ptr a pointer to an integer to get an id for the request
 *
 * \return This functions returns RESULT_OK in case of no error. Following error codes might be returned in cased of
 * 		   failures:
 * 		   - RESULT_NORESOURCES: 			The function call failed due to resource problems.
 * 		   - RESULT_INVALID:				The shared library is not in state AM_API_INITIALIZED_CONNECTED
 */
error_code_t automounter_api_get_snapshot(snapshot_scope_t scope, int *request_id_ptr);

/**
 * \brief returns a string representation of the device state.
 * \param state of the device state
 * \return a string representation of the device state
 */
const char *automounter_api_get_device_state_string(device_state_t state);

/**
 * \brief returns a string representation of the partition state.
 * \param state of the partition state
 * \return a string representation of the partition state
 */
const char *automounter_api_get_partition_state_string(partition_state_t state);

/**
 * \brief returns a string representation of the reason why a partition is unsupported.
 * \param reason the unsupported reason returned from the partition info structure
 * \return a string representation of the reason why a partition is not supported
 */
const char *automounter_api_get_partition_unsupported_reason_string(partition_unsupported_reason_t reason);

#ifdef __cplusplus
}
#endif

#endif
